# finaquant Financial Analytics - www.finaquant.com
# Copyright: Tunc Ali Ktkcoglu 2012, version: 24April2012
# TITLE: Share Class Fee Calculator
# Related web page:
# http://finaquant.com/fee-calculator-and-analyzer-for-share-classes-of-mutual-funds-1-calculation/991
#
# FEE TYPES:
# Front-end load with possible volume discounts
# Back-end load with DCSC (Contingent Deferred Sales Charges)
# Annual distribution fees (12b-1) in #
# Annual management fees with possible volume discounts
# Other annual service fees in # and/or absolute amount
# Other subscription fees in # and/or absolute amount
# Other redemption fees in # and/or absolute amount

#**************************************************************************
# INPUT PARAMETERS
#**************************************************************************
print('Set input parameters')
FrondEndLoadRates = matrix(c(		# percentage and exclusive upper limit
    5.00,        50000,      		# fee rate = 5.00% for 0 <= Amount < 50000
    4.50,        100000,     		# fee rate = 4.50% for 50000 <= Amount < 100000
    3.70,        250000,
    2.80,        500000,
    2.00,        999999999),  		# fee rate = 2% for Amount >= 500000
	ncol=2, byrow=TRUE)
# Set FrondEndLoad = matrix(c(0,999999999),ncol=2,byrow=TRUE) if there is no front-end load	

BackEndLoadCDSCRates = c(
    5.00,               # i.e. 5% deducted if shares are redeemed within 1st year
    4.00,               # i.e. 4% deducted if shares are redeemed within 2nd year
    3.00,
    2.00,
    1.00 )           	# no deductions after 6th year
# Set BackEndLoadCDSC = c(0) if there is no back-end load with CDSC

AnnualManagementFeeRates = matrix(c(   	# fee rates in BP (basis points, 1 BP = 0.01% = 1/100000)
    80,          50000,         		# fee rate = 80 BP for 0 <= Amount < 500000
    70,          100000,        		# fee rate = 70 BP for 500000 <= Amount < 1000000
    60,          999999999),    		# fee rate = 60 BP for Amount >= 1000000
	ncol=2, byrow=TRUE)
# Set ManagementFeeRates = matrix(c(x,999999999),ncol=2,byrow=TRUE) if there is a single
# fee rate without volume discounts (i.e. tiered rates)

AnnualDistributionFeeRate = 0.5; 	# i.e. 0.5% annually

# Fee scale calculation logic for management fees
MngFeeScaleLogic = 'class'
# class: There is a single fee rate applied on the whole volume
# f.e. volume = $600000 --> MngFee = $600000 x 70/10000 w.r.t. given
# tiered rates above in AnnualManagementFeeRates
#
# level: Each volume range is multiplied with its corresponding fee rate
# f.e. volume = $600000 --> MngFee = $500000 x 80/10000 + $100000 x 70/10000

OtherAnnualServiceFeeRates = c(10, 0) 	# 10 BP + $0 per share, annually 

OtherSubscriptionFeeRates = c(0, 0)   	# 0% + $0 per share, at subscription

OtherRedemptionFeeRates = c(0, 0)      	# 0% + $0 per share, at redemption
    
# Total number of periods (years) for calculation
N = 20

# Assumed fund return for all years before all fee costs
AssumedFixedFundReturn = 5     # i.e. 5%

# Fund returns per each year within the calculation time span (i.e. N years)
AssumedFundReturns = AssumedFixedFundReturn * rep(1,N)
# set AssumedFundReturns = c(x1, x2, x3, ... xN) if you want to give assumed
# fund returns for each period (year) seperately

# Number of fund shares purchased (#subscriptions at the beginning)
NumberOfShares = 500

# Gross purchase price of a share before all costs at the beginning 
SharePrice = 100

#**************************************************************************
# all input parameters are ready: call multi-period fee function
#**************************************************************************

# load functions
source('FeeCalculator_functions.r')

# TEST START
vec = FeeCalculator_SinglePeriod( 
    FrondEndLoadRates,
    BackEndLoadCDSCRates,
    AnnualDistributionFeeRate,
    AnnualManagementFeeRates,
    MngFeeScaleLogic,
    OtherAnnualServiceFeeRates,
    OtherSubscriptionFeeRates,
    OtherRedemptionFeeRates,
    AssumedFundReturns[2],
    60000,
    NumberOfShares,     			
    2,                 
    N) 
display_variable(vec, 'TEST vec')
# TEST END

print('Call multi-period fee calculation function')
res = FeeCalculator_MultiPeriod(    
	FrondEndLoadRates,              
    BackEndLoadCDSCRates,           
    AnnualDistributionFeeRate,      
    AnnualManagementFeeRates,       
    MngFeeScaleLogic,               
    OtherAnnualServiceFeeRates,     
    OtherSubscriptionFeeRates,      
    OtherRedemptionFeeRates,        
    AssumedFundReturns,             
    SharePrice,       
    NumberOfShares,                   
    N);

ResultMatrix = res[[1]]
ResultMatrixFields = res[[2]]
EffectiveFundReturns = res[[3]]
EffectiveAssetValue = res[[4]]

# total returns as percentage
InitialInvestment = NumberOfShares * SharePrice;
TotalFundReturns = (EffectiveAssetValue - InitialInvestment) / InitialInvestment * 100;

# construct result table with total returns and final asset values at each period-end
ResultTable = t(data.frame(row.names = ResultMatrixFields, t(ResultMatrix)))
AddedTable = data.frame(EffReturnPeriod=EffectiveFundReturns, TotalReturn=TotalFundReturns, FinalAssetValue=EffectiveAssetValue)
ResultTable = cbind(ResultTable, AddedTable)

#data.frame(EffReturnPeriod=EffectiveFundReturns, TotalReturn=TotalFundReturns, FinalAssetValue=EffectiveAssetValue));

#**************************************************************************
# write results into an excel file in the same working directory
#**************************************************************************
print('Write results into an excel sheet')
require(XLConnect) #load the package

# write data frame into excel sheet
wbFilename = "FeeResults_R.xlsx"
sheet = "FeeResuts"
dataName = "fees"
if(file.exists(wbFilename)) file.remove(wbFilename)
writeNamedRegionToFile(wbFilename, data = ResultTable, name = dataName,
	formula = paste(sheet, "$A$1", sep = "!"),header = TRUE)
